﻿/**
 * Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
 *
 * This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
 * confidential. It may not be copied and/or distributed without the express permission of 
 * "Infront Financial Technology GmbH".
 */
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using Vwd.Middleware.Common.DataModel.Base;
using Vwd.Middleware.Common.DataModel.Base.AfuData;
using Vwd.Middleware.Common.DataModel.Base.AfuData.ComplexTypes;
using Vwd.Middleware.Common.DataModel.Base.BaseClasses;
using ErrorCodes = Vwd.Middleware.Common.DataModel.Base.Enums.ErrorCodes;
using MessageSerializer = Gevasys.DataCore.RequestData.ProtoBuf.V2.MessageSerializer;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * AfuCompanyDataSample
    * returns AFU DATA for given symbols
    * it is possible to get all data or only data from the given categories (StatementCategories)
    */
    public class AfuCompanyDataSample
    {
        public void TestAfuCompanyData(RoutingDataStore rds, MessageSerializer serializer)
        {
            // define parameters to use within the request
            var symbols = new List<string>
            {
                "DE0006231004",
                "US29275Y1029"
            };

            // create the request object
            AfuCompanyDataRequest request = new AfuCompanyDataRequest
            {
                Header = new Header()
                {
                    User = Authentication.GetUser(),
                    Password = Authentication.GetPassword()
                },
                SymbolList = symbols,
                WithShareholder = true,
                WithEmployees = true,
                WithStatements = true,
                // OPTIONAL filter by category type. Only categories referenced in this list will be delivered in the response
                StatementCategories = new List<AfuCategoryType>() { AfuCategoryType.Benchmark }
            };

            // call the DMC backend
            Console.WriteLine("Calling AfuCompanyData function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.GENERAL_AFU_COMPANY_DATA,
                      InputFormat.PROTO.ToString(),
                      OutputFormat.PROTO.ToString(),
                      serializer.Serialize(OutputFormat.PROTO, request));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            AfuCompanyDataResponse responseMessage = serializer.Deserialize<AfuCompanyDataResponse>(InputFormat.PROTO, response.GetData());

            if (responseMessage.HasError)
            {
                Logger.Log.Warn("Error: " + responseMessage.Error.ErrorMessage);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }


        private static void WriteResultsToConsole(AfuCompanyDataResponse companyDataResponse)
        {
            foreach (CompanyDataResponseItem companyDataResponseItem in companyDataResponse.Results)
            {
                if (companyDataResponseItem.Error != null && companyDataResponseItem.Error.ErrorCode != ErrorCodes.NoError)
                {
                    Logger.Log.Error("An error ocurred for company data. Error message: " + companyDataResponseItem.Error.ErrorMessage);
                    return;
                }

                Console.WriteLine();
                Console.WriteLine("*******************" + companyDataResponseItem.Symbol + "*******************");
                CompanyData companyData = companyDataResponseItem.CompanyData;
                AfuCompanyBaseData baseData = companyData.BaseData;
                if (baseData != null)
                {
                    Console.WriteLine("\n################## BaseData ##################\n");
                    Console.WriteLine("Adress: " + baseData.Adress);
                    Console.WriteLine("Copyright: " + baseData.Copyright);
                    Console.WriteLine("CurrencyIso: " + baseData.CurrencyIso);
                    Console.WriteLine("DataChangeDate: " + baseData.DataChangeDate);
                    Console.WriteLine("Email: " + baseData.Email);
                    Console.WriteLine("Fax: " + baseData.Fax);
                    Console.WriteLine("FiscalDay: " + baseData.FiscalDay);
                    Console.WriteLine("FiscalMonth: " + baseData.FiscalMonth);
                    Console.WriteLine("FoundationDate: " + baseData.FoundationDate);
                    Console.WriteLine("IndustrialSector: " + baseData.IndustrialSector);
                    Console.WriteLine("Isin: " + baseData.Isin);
                    Console.WriteLine("MethodOfFinancing: " + baseData.MethodOfFinancing);
                    Console.WriteLine("Name: " + baseData.Name);
                    Console.WriteLine("Notefinancials: " + baseData.Notefinancials);
                    Console.WriteLine("ParValue: " + baseData.ParValue);
                    Console.WriteLine("Phone: " + baseData.Phone);
                    Console.WriteLine("Profile: " + baseData.Profile);
                    Console.WriteLine("Sector: " + baseData.Sector);
                    Console.WriteLine("VendorlastUpdate: " + baseData.VendorlastUpdate);
                    Console.WriteLine("Website: " + baseData.Website);
                }

                if (companyData.Employees != null)
                {
                    Console.WriteLine("\n################## Employees ##################");
                    foreach (CompanyEmployeeItem employee in companyData.Employees)
                    {
                        Console.WriteLine("Name: " + employee.Name + ", Job: " + employee.Job + ", IsChairPerson: " + employee.IsChairPerson + ", DataChangeDate: " + employee.DataChangeDate);
                    }
                }

                if (companyData.Shareholders != null)
                {
                    Console.WriteLine("\n################## Shareholders ##################");
                    foreach (CompanyShareholderItem shareholderItem in companyData.Shareholders)
                    {
                        Console.WriteLine("ShareholderName: " + shareholderItem.ShareholderName + ", Percentage: " + shareholderItem.Percentage + ", DataChangeDate: " + shareholderItem.DataChangeDate);
                    }
                }

                Console.WriteLine("\n################## Statements by Categorie ##################");
                if (companyData.StatementsByCategory != null)
                {
                    List<StatementByCategoryItem> statementsByCategory = companyData.StatementsByCategory;
                    foreach (StatementByCategoryItem statement in statementsByCategory)
                    {
                        String statementType = "";
                        AfuCategoryType afuCategoryType = statement.CategoryType;
                        List<CompanyStatementItem> companyStatementItems = statement.CompanyStatements;
                        Console.WriteLine("\n------------------" + afuCategoryType + "------------------");
                        foreach (CompanyStatementItem companyStatementItem in companyStatementItems)
                        {
                            if (!statementType.Equals(companyStatementItem.StatementType))
                            {
                                statementType = companyStatementItem.StatementType;

                                Console.WriteLine("\n" + statementType + " : " + companyStatementItem.StatementTypeDescription + "\n");
                            }
                            Console.WriteLine("Value : " + companyStatementItem.Value + ", Year : " + companyStatementItem.Year + ", DataChangeDate : " + companyStatementItem.DataChangeDate);
                        }
                    }
                }
            }
        }
    }
}
