﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.RequestData.ProtoBuf.V2.CorporateActions;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * CorporateActionsVwdCodeSample 
    * - get all corporate Actions to a vwdCode in the given period
    * - parameters to be set: vwd code, start, end and flag WithFactorizedDividends (optional)
    */
    public class CorporateActionsVwdCodeSample
    {
        public void TestCorporateActionsVwdCode(RoutingDataStore rds, MessageSerializer serializer)
        {
            // create the request object
            CorporateActionsRequestMessage corporateActionsRequest = new CorporateActionsRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                start = (long)new DateTime(2015, 1, 1, 0, 0, 0).Subtract(DateTime.MinValue.AddYears(1969)).TotalMilliseconds,
                end = (long)DateTime.Now.Subtract(DateTime.MinValue.AddYears(1969)).TotalMilliseconds,    
                vwdCode = "DB1_D.BXE4",
                withFactorizedDividends = false
            };

            // call the DMC backend
            Console.WriteLine("Calling CorporateActions function. Please, note that specific permissions might be needed by contract.");
            Console.WriteLine("Call Request-CORPORATE_ACTIONS for vwdode: " + corporateActionsRequest.vwdCode);
            var response = rds.CallFunction(StaticDataFunctions.CORPORATE_ACTIONS,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize(OutputFormat.PROTO, corporateActionsRequest));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            CorporateActionsMessage responseMessage = serializer.Deserialize<CorporateActionsMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private static void WriteResultsToConsole(CorporateActionsMessage responseMessage)
        {
            if (responseMessage.actions != null)
            {
                Console.WriteLine(responseMessage.vwdCode);
                foreach (CorporateActionMessage actionMessage in responseMessage.actions)
                {
                    Console.WriteLine("value: " + actionMessage.value + ",type: " + actionMessage.type + ",currency: " + actionMessage.currency + ",dateTime: " + new DateTime(1970, 1, 1).AddMilliseconds(actionMessage.dateTime));
                }
            }
        }
    }
}
