﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;


namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * FundHistorySample
    * returns historical Data for an instrument (fund) for the time from "dateFrom" to "dateTo"
    */
    public class FundHistorySample
    {
        public void TestFundHistory(RoutingDataStore rds, MessageSerializer serializer)
        {
            // define parameters for request
            DateTime from = new DateTime(2020, 03, 01, 9, 0, 0);
            DateTime to = new DateTime(2020, 06, 19, 13, 00, 0);
            String instrument = "DE000A0X7541.FONDS.EUR";

            // create the request object
            HistoricalTimeSeriesRequestMessage requestMessage = new HistoricalTimeSeriesRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                aggregation = HistoricAggregationParam.Day,
                instrument = instrument,
                withDividens = false,
                withCorporateActions = true,
                tickType = TickTypeParam.Trade,
                dateFrom = DateTimeConverter.Convert(ref from),
                dateTo = DateTimeConverter.Convert(ref to),
                currency = "EUR"
            };

            // call the DMC backend
            Console.WriteLine("Calling HistoricFUndQuoteData function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.FUND_HISTORY,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<HistoricalTimeSeriesRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            HistoricalFundQuoteDataMessage responseMessage = serializer.Deserialize<HistoricalFundQuoteDataMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private void WriteResultsToConsole(HistoricalFundQuoteDataMessage responseMessage)
        {
            // write results to console
            Console.WriteLine("----------------" + responseMessage.instrument + "----------------");
            foreach (HistoricalFundQuoteDataEntry entry in responseMessage.entry)
            {
                Console.WriteLine("issue: " + entry.issue +
                    " ,netAssetValue: " + entry.netAssetValue +
                    " ,redemption: " + entry.redemption +
                    " ,time: " + ConvertToDateTime((double)entry.time));
            }
        }

        public DateTime ConvertToDateTime(double dateAsLong)
        {
            System.DateTime date = new DateTime(1970, 1, 1, 0, 0, 0, 0, System.DateTimeKind.Utc);
            date = date.AddMilliseconds(dateAsLong).ToLocalTime();
            return date;
        }
    }
}
