﻿/**
 * Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
 *
 * This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
 * confidential. It may not be copied and/or distributed without the express permission of 
 * "Infront Financial Technology GmbH".
 */
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.RequestData.ProtoBuf.V2.DMXML;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * MarketSnapshotSample
    * returns all instruments from the given market
    * [PT, HKI, CM, BL, CHIX, FR, BUBA, N, Q, FXVWD, NIKKEI, PHIL, NL, CB]
    */
    public class MarketSnapshotSample
    {
        public void TestMarketSnapshot(RoutingDataStore rds, MessageSerializer serializer)
        {          
            // create the request object
            MarketSnapshotRequestMessage requestMessage = new MarketSnapshotRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword(),
                    authToken = rds.GetAuthToken()
                },
                marketCode = "NIKKEI"             
            };

            // call the DMC backend
            Console.WriteLine("Calling MarketSnapshot function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.MARKET_SNAPSHOT_V3, 
                InputFormat.PROTO.ToString(), 
                OutputFormat.PROTO.ToString(),                               
                serializer.Serialize<MarketSnapshotRequestMessage>(OutputFormat.PROTO, requestMessage));
            
            Logger.Log.Info("RoutingDataStore response for MarketSnapshotRequestMessage: " + requestMessage.marketCode);
            Console.WriteLine("RoutingDataStore response for MarketSnapshotRequestMessage (QuotationDataListsMessage): " + requestMessage.marketCode);

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            QuotationDataListsMessage responseMessage = serializer.Deserialize<QuotationDataListsMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            if (responseMessage.quotationList == null)
            {
                Logger.Log.Warn("Quotations list is empty.Please, check if the requested 'marketCode' is correct.");
            }
            else
            {
                PrintInstrumentsFields(responseMessage.quotationList);
            }
        }

        private void PrintInstrumentsFields(List<QuotationDataListMessage> quotationList)
        {
            foreach(QuotationDataListMessage quotationListMessage in quotationList)
            {
                string vwdCode = quotationListMessage.vwdCode;

                Console.WriteLine("VWD Code: " + vwdCode + "\r\n");

                GetQuotationDataAndPrint(quotationListMessage);               
            }            
        }

        private void GetQuotationDataAndPrint(QuotationDataListMessage quotationListMessage)
        {
            foreach(QuotationDataEntryMessage quotationDataEntry in quotationListMessage.fieldList)
            {
                QuotationDataEntryMessage fieldForInstrument = quotationDataEntry;

                string message = "ID: " + fieldForInstrument.id + " || " +
                                 "Field Type: " + fieldForInstrument.fieldType + " || ";

                switch (fieldForInstrument.fieldType)
                {
                    case QuotationDataFieldType.DT_BOOLEAN:
                        bool? booleanValue = fieldForInstrument.boolValue;
                        if (booleanValue != null)
                            message += "Value: " + booleanValue.Value.ToString() + "\r\n";
                        break;
                    case QuotationDataFieldType.DT_DOUBLE:
                        double? doubleValue = fieldForInstrument.doubleValue;
                        message += "Value: " + doubleValue.Value.ToString() + "\r\n";
                        break;
                    case QuotationDataFieldType.DT_LONG:
                        long? longValue = fieldForInstrument.longValue;
                        if (longValue != null)
                            message += "Value: " + longValue.Value.ToString() + "\r\n";
                        break;
                    case QuotationDataFieldType.DT_INT:
                        int? intValue = fieldForInstrument.intValue;
                        if (intValue != null)
                            message += "Value: " + intValue.Value.ToString() + "\r\n";
                        break;
                    case QuotationDataFieldType.DT_STRING:
                        String stringValue = fieldForInstrument.stringValue;
                        message += " Value: " + stringValue + "\r\n";
                        break;
                    case QuotationDataFieldType.DT_DATETIME:
                        long? dateTimeValue = fieldForInstrument.dateTimeValue;
                        if (dateTimeValue.HasValue)
                            message += " Value: " + DateTimeConverter.FromUtcMillis(dateTimeValue.Value) + "\r\n";
                        break;
                    default:
                        break;
                }

                Console.WriteLine(message);
            }
        }
    }
}
