﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using Vwd.Middleware.Common.DataModel.Base;
using Vwd.Middleware.Common.DataModel.Base.BaseClasses;
using Vwd.Middleware.Common.DataModel.Base.ComplexTypes;
using MessageSerializer = Gevasys.DataCore.RequestData.ProtoBuf.V2.MessageSerializer;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * EventCalendarSample
    * returns events for given symbol (and time period). 
    */
    public class EventCalendarSample
    {
        public void TestEventCalendar(RoutingDataStore rds, MessageSerializer serializer)
        {
            // define parameters to use within the request
            List<string> symbols = new List<string> { "DE0006231004", "US29275Y1029" };
            DateTime dateFrom = new DateTime(2018, 01, 01);
            string culture = "en-EN";

            // create the request object
            EventsRequest request = new EventsRequest
            {
                Header = new Header()
                {
                    User = Authentication.GetUser(),
                    Password = Authentication.GetPassword()
                },
                SymbolList = symbols,
                DateFrom = Timestamp.ConvertToTimestamp(dateFrom), // is the beginn from the requested timeperiod until now, if the date is null the period is the last 6 months                
                Culture = culture
            };

            // call the DMC backend
            Console.WriteLine("Calling Events function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.GENERAL_EVENTS,
                  InputFormat.PROTO.ToString(),
                  OutputFormat.PROTO.ToString(),
                  serializer.Serialize<EventsRequest>(OutputFormat.PROTO, request));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            EventsResponse responseMessage = serializer.Deserialize<EventsResponse>(InputFormat.PROTO, response.GetData());
            
            if (responseMessage.HasError)
            {
                Logger.Log.Warn("Error: " + responseMessage.Error.ErrorMessage);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private static void WriteResultsToConsole(EventsResponse responseMessage)
        {
            // print results to console
            Console.WriteLine("Culture - " + responseMessage.Culture);

            foreach (EventListItem events in responseMessage.Results)
            {
                if (events.HasError)
                {
                    Logger.Log.Error("There was an error in events results: " + events.Error.ErrorMessage);
                }
                else
                {
                    Console.WriteLine("------------------------ Instrument Data ------------------------");
                    Console.WriteLine("ID: " + events.Instrument.Iid);
                    Console.WriteLine("ISIN: " + events.Instrument.Isin);
                    Console.WriteLine("Name: " + events.Instrument.Name);
                    Console.WriteLine("Name short: " + events.Instrument.NameShort);
                    Console.WriteLine("Ticker Symbol: " + events.Instrument.TickerSymbol);
                    Console.WriteLine("Type: " + events.Instrument.Type);
                    Console.WriteLine("Wkn: " + events.Instrument.Wkn);

                    Console.WriteLine("--- Events list: ---");
                    List<EventItem> eventListItem = events.EventList;
                    if (eventListItem != null)
                    {
                        foreach (EventItem eventItem in eventListItem)
                        {
                            Console.WriteLine("Title: " + eventItem.LocalizedTitle + " ,Date: " + eventItem.Date);
                        }
                    }
                }
            }
        }
    }
}
