﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.RequestData.ProtoBuf.V2.Search;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
     * FundFinderSample
     * returns funds
     * count gives the maximum for the requested funds
     * with the query the result can be filtered, without the query all funds will be find
     * 
     */
    public class FundFinderSample
    {
        public void TestFundFinder(RoutingDataStore rds, MessageSerializer serializer)
        {
            // create the request object
            FNDFinderRequestMessage request = new FNDFinderRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                count = 10,
                //optional query
                query = "issuername==Deka && fundtype==Aktienfonds&&investmentFocus==Aktien Europa"
            };

            // call the DMC backend
            Console.WriteLine("Calling FundFinder function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.FUND_FINDER,
                InputFormat.PROTO.ToString(), 
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<FNDFinderRequestMessage>(OutputFormat.PROTO, request));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            FNDFinderMessage responseMessage = serializer.Deserialize<FNDFinderMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            // get results list and print in console
            WriteResultsToConsole(responseMessage);                
        }

        private void WriteResultsToConsole(FNDFinderMessage responseMessage)
        {
            List<FinderFNDListMessage> finderList = responseMessage.finderList;
            foreach (FinderFNDListMessage finderFNDListMessage in finderList)
            {
                Console.WriteLine("\n**************InstrumentData**************");
                GeneralSearchInstrumentDataMessage instrumentData = finderFNDListMessage.instrumentData;
                Console.WriteLine("Name: " + instrumentData.name);
                Console.WriteLine("WKN: " + instrumentData.wkn);
                Console.WriteLine("ISIN: " + instrumentData.isin);
                Console.WriteLine("Type: " + instrumentData.type);
                Console.WriteLine("IID: " + instrumentData.iid);

                Console.WriteLine("\n**************BaseData**************");
                Console.WriteLine("fundtype: " + finderFNDListMessage.fundtype);
                Console.WriteLine("fundtypeBviCoarse: " + finderFNDListMessage.fundtypeBviCoarse);
                Console.WriteLine("fundVolume: " + finderFNDListMessage.fundVolume);
                Console.WriteLine("investmentFocus: " + finderFNDListMessage.investmentFocus);
                Console.WriteLine("name: " + finderFNDListMessage.name);
                Console.WriteLine("priceDataType: " + finderFNDListMessage.priceDataType);
                Console.WriteLine("country: " + finderFNDListMessage.country);
                Console.WriteLine("region: " + finderFNDListMessage.region);
                Console.WriteLine("distributionStrategy: " + finderFNDListMessage.distributionStrategy);
                Console.WriteLine("etf: " + finderFNDListMessage.etf);
                Console.WriteLine("marketAdmission: " + finderFNDListMessage.marketAdmission);

                Console.WriteLine("\n**************PriceData**************");
                Console.WriteLine("price: " + finderFNDListMessage.price);
                Console.WriteLine("accountFee: " + finderFNDListMessage.accountFee);
                Console.WriteLine("managementFee: " + finderFNDListMessage.managementFee);
                Console.WriteLine("changeNet: " + finderFNDListMessage.changeNet);
                Console.WriteLine("changePercent: " + finderFNDListMessage.changePercent);
                Console.WriteLine("date: " + finderFNDListMessage.date);
                Console.WriteLine("maximumLoss3y: " + finderFNDListMessage.maximumLoss3y);
                Console.WriteLine("ratingFeri: " + finderFNDListMessage.ratingFeri);
                Console.WriteLine("ratingMorningstar: " + finderFNDListMessage.ratingMorningstar);
                Console.WriteLine("repurchasingPrice: " + finderFNDListMessage.repurchasingPrice);
                Console.WriteLine("riskclassAttrax: " + finderFNDListMessage.riskclassAttrax);
                Console.WriteLine("ter: " + finderFNDListMessage.ter);

                Console.WriteLine("\n**************QuoteData**************");
                GeneralSearchQuoteDataMessage quoteData = finderFNDListMessage.quoteData;
                Console.WriteLine("vwdCode: " + quoteData.vwdCode);
                Console.WriteLine("marketName: " + quoteData.marketName);
                Console.WriteLine("marketVwd: " + quoteData.marketVwd);
                Console.WriteLine("currencyIso: " + quoteData.currencyIso);
                Console.WriteLine("mmwkn: " + quoteData.mmwkn);
                Console.WriteLine("qid: " + quoteData.qid);

                Console.WriteLine("\n**************Issue**************");
                Console.WriteLine("issueDate: " + finderFNDListMessage.issueDate);
                Console.WriteLine("issuePrice: " + finderFNDListMessage.issuePrice);
                Console.WriteLine("issuername: " + finderFNDListMessage.issuername);
                Console.WriteLine("issueSurcharge: " + finderFNDListMessage.issueSurcharge);

                Console.WriteLine("\n**************BenchMark**************");
                Console.WriteLine("Outperformance1y: " + finderFNDListMessage.benchmarkOutperformance1y);
                Console.WriteLine("Outperformance3y: " + finderFNDListMessage.benchmarkOutperformance3y);

                Console.WriteLine("\n**************BviPerformance**************");
                Console.WriteLine("CurrentYear: " + finderFNDListMessage.bviperformanceCurrentYear);
                Console.WriteLine("1d: " + finderFNDListMessage.bviperformance1d);
                Console.WriteLine("1w: " + finderFNDListMessage.bviperformance1w);
                Console.WriteLine("1m: " + finderFNDListMessage.bviperformance1m);
                Console.WriteLine("3m: " + finderFNDListMessage.bviperformance3m);
                Console.WriteLine("6m: " + finderFNDListMessage.bviperformance6m);
                Console.WriteLine("1y: " + finderFNDListMessage.bviperformance1y);
                Console.WriteLine("3y: " + finderFNDListMessage.bviperformance3y);
                Console.WriteLine("5y: " + finderFNDListMessage.bviperformance5y);
                Console.WriteLine("10y: " + finderFNDListMessage.bviperformance10y);

                Console.WriteLine("\n**************SharpRatio**************");
                Console.WriteLine("1w: " + finderFNDListMessage.sharpeRatio1w);
                Console.WriteLine("1m: " + finderFNDListMessage.sharpeRatio1m);
                Console.WriteLine("3m: " + finderFNDListMessage.sharpeRatio3m);
                Console.WriteLine("6m: " + finderFNDListMessage.sharpeRatio6m);
                Console.WriteLine("1y: " + finderFNDListMessage.sharpeRatio1y);
                Console.WriteLine("3y: " + finderFNDListMessage.sharpeRatio3y);
                Console.WriteLine("5y: " + finderFNDListMessage.sharpeRatio5y);
                Console.WriteLine("10y: " + finderFNDListMessage.sharpeRatio10y);

                Console.WriteLine("\n**************Volatility**************");
                Console.WriteLine("1w: " + finderFNDListMessage.volatility1w);
                Console.WriteLine("1m: " + finderFNDListMessage.volatility1m);
                Console.WriteLine("3m: " + finderFNDListMessage.volatility3m);
                Console.WriteLine("6m: " + finderFNDListMessage.volatility6m);
                Console.WriteLine("1y: " + finderFNDListMessage.volatility1y);
                Console.WriteLine("3y: " + finderFNDListMessage.volatility3y);
                Console.WriteLine("5y: " + finderFNDListMessage.volatility5y);
                Console.WriteLine("10y: " + finderFNDListMessage.volatility10y);
            }
        }
    }
}
