﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.RequestData.ProtoBuf.V2.Search;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * FundStaticDataSample
    * returns static data from given fund
    * 
    */
    public class FundStaticDataSample
    {
        public void TestFundStaticData(RoutingDataStore rds, MessageSerializer serializer)
        {
            // create the request object
            StaticDataRequestMessage request = new StaticDataRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                market = "Fonds",
                symbol = "LU0106235293",
                providerPreference = ProviderPreference.VWD
            };

            // call the DMC backend
            Console.WriteLine("Calling Fund function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.FUND,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<StaticDataRequestMessage>(OutputFormat.PROTO, request));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            FNDStaticDataMessage responseMessage = serializer.Deserialize<FNDStaticDataMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteInstrumentDataToConsole(responseMessage);

            WriteQuoteDataToConsole(responseMessage);

            WriteBaseDataToConsole(responseMessage);

            WriteIssueDataToConsole(responseMessage);

            // gets benchmark instrumentData and quoteData and print in console
            WriteBenchmarkResultsToConsole(responseMessage);
        }

        private static void WriteInstrumentDataToConsole(FNDStaticDataMessage responseMessage)
        {
            GeneralSearchInstrumentDataMessage instrumentData = responseMessage.instrumentData;
            if (instrumentData != null)
            {
                Console.WriteLine("\n-----------------**    InstrumentData     **-----------------");
                Console.WriteLine("name: " + instrumentData.name);
                Console.WriteLine("wkn: " + instrumentData.wkn);
                Console.WriteLine("isin: " + instrumentData.isin);
                Console.WriteLine("type: " + instrumentData.type);
                Console.WriteLine("iid: " + instrumentData.iid);
            }
        }

        private static void WriteQuoteDataToConsole(FNDStaticDataMessage responseMessage)
        {
            // get quoteData and print in console
            GeneralSearchQuoteDataMessage quoteData = responseMessage.quoteData;
            if (quoteData != null)
            {
                Console.WriteLine("\n-----------------**    QuoteData     **-----------------");
                Console.WriteLine("vwdCode: " + quoteData.vwdCode);
                Console.WriteLine("marketName: " + quoteData.marketName);
                Console.WriteLine("marketVwd: " + quoteData.marketVwd);
                Console.WriteLine("currencyIso: " + quoteData.currencyIso);
                Console.WriteLine("mmwkn: " + quoteData.mmwkn);
                Console.WriteLine("qid: " + quoteData.qid);
            }
        }

        private static void WriteBaseDataToConsole(FNDStaticDataMessage responseMessage)
        {
            Console.WriteLine("\n-----------------**    BaseData     **-----------------");
            Console.WriteLine("country: " + responseMessage.country);
            Console.WriteLine("currency: " + responseMessage.currency);
            Console.WriteLine("marketAdmission: " + responseMessage.marketAdmission);
            Console.WriteLine("\nstrategy: " + responseMessage.strategy + "\n");
            Console.WriteLine("fundManager: " + responseMessage.fundManager);
            Console.WriteLine("managementFee: " + responseMessage.managementFee);
            Console.WriteLine("distributionStrategy: " + responseMessage.distributionStrategy);
        }

        private void WriteIssueDataToConsole(FNDStaticDataMessage responseMessage)
        {
            Console.WriteLine("\n-----------------**    IssueData     **-----------------");
            Console.WriteLine("issueDate(UTC): " + ConvertToDateTime((double)responseMessage.issueDate));
            Console.WriteLine("issueSurcharge: " + responseMessage.issueSurcharge);
            Console.WriteLine("issuerCity: " + responseMessage.issuerCity);
            Console.WriteLine("issuerName: " + responseMessage.issuerName);
            Console.WriteLine("issuerPhone: " + responseMessage.issuerPhone);
            Console.WriteLine("issuerPostalcode: " + responseMessage.issuerPostalcode);
            Console.WriteLine("issuerStreet: " + responseMessage.issuerStreet);
            Console.WriteLine("issuerUrl: " + responseMessage.issuerUrl);
        }

        private DateTime ConvertToDateTime(double dateAsLong)
        {
            DateTime date = new DateTime(1970, 1, 1, 0, 0, 0, 0, DateTimeKind.Utc);
            date = date.AddMilliseconds(dateAsLong).ToLocalTime();
            return date;
        }

        private static void WriteBenchmarkResultsToConsole(FNDStaticDataMessage responseMessage)
        {
            IdentifierDataMessage benchmark = responseMessage.benchmark;
            if (benchmark != null)
            {
                Console.WriteLine("\n-----------------**    Benchmark     **-----------------");
                Console.WriteLine(responseMessage.benchmarkName);
                GeneralSearchInstrumentDataMessage benchmarkInstrumentData = benchmark.instrumentData;
                if (benchmarkInstrumentData != null)
                {
                    Console.WriteLine("-----------------Benchmark - InstumentData-----------------");
                    Console.WriteLine("name: " + benchmarkInstrumentData.name);
                    Console.WriteLine("wkn: " + benchmarkInstrumentData.wkn);
                    Console.WriteLine("isin: " + benchmarkInstrumentData.isin);
                    Console.WriteLine("type: " + benchmarkInstrumentData.type);
                    Console.WriteLine("iid: " + benchmarkInstrumentData.iid);

                }
                GeneralSearchQuoteDataMessage benchmarkQuoteData = benchmark.quoteData;
                if (benchmarkQuoteData != null)
                {
                    Console.WriteLine("\n-----------------Benchmark - QuoteData-----------------");
                    Console.WriteLine("vwdCode: " + benchmarkQuoteData.vwdCode);
                    Console.WriteLine("marketName: " + benchmarkQuoteData.marketName);
                    Console.WriteLine("marketVwd: " + benchmarkQuoteData.marketVwd);
                    Console.WriteLine("currencyIso: " + benchmarkQuoteData.currencyIso);
                    Console.WriteLine("mmwkn: " + benchmarkQuoteData.mmwkn);
                    Console.WriteLine("qid: " + benchmarkQuoteData.qid);
                }
            }
        }
    }
}
