﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * GeneralNewsSearchSample
    * retrieves news via paging.  
    */
    public class GeneralNewsSearchSample
    {
        public void TestGeneralNewsSearch(RoutingDataStore rds, MessageSerializer serializer)
        {
            // Define parameters to use in request
            DateTime from_news = DateTime.Now.AddHours(-1);
            DateTime to_news = DateTime.Now;
            uint newsItemsPerPage = 15; // has to be a value in [2,300]       
            uint pagingOffset = 0;

            // create the request object
            GeneralNewsSearchRequestMessage requestMessage = new GeneralNewsSearchRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                dateFrom = DateTimeConverter.Convert(ref from_news),
                dateTo = DateTimeConverter.Convert(ref to_news),
                withText = true,
                searchCount = newsItemsPerPage,               
                offset = pagingOffset
            };

            // initialize variables to use in the loop for paging
            bool moreNewsAvailable = true;
            int? totalNumberAvailableNews = null;

            while (moreNewsAvailable) // for paging.
            {
                // update offset with number of news processed in previous page (newsItemsPerPage)
                // to get news in next page
                requestMessage.offset = pagingOffset;

                // call the DMC backend to get news
                Console.WriteLine("Calling NewsSearch function. Please, note that specific permissions might be needed by contract.");
                StaticDataResponse response = rds.CallFunction(StaticDataFunctions.GENERAL_NEWSSEARCH,
                    InputFormat.PROTO.ToString(), 
                    OutputFormat.PROTO.ToString(),
                    serializer.Serialize<GeneralNewsSearchRequestMessage>(OutputFormat.PROTO, requestMessage));             

                if (response.GetData() == null)
                {
                    Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                    Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                    return;
                }

                //get results
                NewsListMessage responseMessage = serializer.Deserialize<NewsListMessage>(InputFormat.PROTO, response.GetData());

                if (responseMessage.header.error_code != ErrorCodes.Ok)
                {
                    Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                    Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                    return;
                }

                //this step is only done on the first loop iteration, when totalCount has no value assigned
                if (totalNumberAvailableNews == null) {
                    if (responseMessage.totalCount.HasValue)
                    {
                        totalNumberAvailableNews = responseMessage.totalCount.Value;
                    }
                    else
                    {
                        Logger.Log.Error("TotalCount of news is empty in response");
                        break;
                    }
                }
                         
                // get new and print in console
                foreach (var newsItem in responseMessage.news)
                {
                    DateTime newTime = newsItem.time.HasValue
                        ? DateTimeConverter.FromUtcMillis(newsItem.time.Value)
                        : DateTime.MinValue;
                    Console.WriteLine("id: " + newsItem.Id + "; Headline " + newsItem.Headline + "; Text: " +
                                      newsItem.Body + "; Date: " + newTime.ToString());
                }

                pagingOffset += newsItemsPerPage;
                //when all news are printed, loop is finished (hasMoreNews = false)
                moreNewsAvailable = pagingOffset < totalNumberAvailableNews;
            }
        }
    }
}
