﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * HistoricalTimeSeriesSample
    * returns historical Data for an instrument (OHLCV) for the requested time period 
    */
    public class HistoricalTimeSeriesSample
    {
        /// <summary>
        /// TestHistoricalTimeSeries
        /// returns historical Data for an instrument (OHLCV) for the requested time period
        /// </summary>
        public void TestHistoricalTimeSeries(RoutingDataStore rds, MessageSerializer serializer)
        {
            // define parameters to use within the request
            DateTime startDate = new DateTime(2020, 01, 03, 12, 0, 0);
            DateTime endDate = new DateTime(2020, 01, 15, 14, 00, 0);
            String instrument = "LU0159196392.FONDS.EUR";

            // create the request object
            HistoricalTimeSeriesRequestMessage requestMessage = new HistoricalTimeSeriesRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                aggregation = HistoricAggregationParam.Day,
                instrument = instrument,
                withDividens = false,
                withCorporateActions = false,
                tickType = TickTypeParam.Trade,
                dateFrom = DateTimeConverter.Convert(ref startDate),
                dateTo = DateTimeConverter.Convert(ref endDate),
                // new Intraday parameter
                withIntraday = true
            };

            // call the DMC backend
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.HISTORIC_DATA_V2,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<HistoricalTimeSeriesRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            HistoricalTimeSeriesMessage responseMessage = serializer.Deserialize<HistoricalTimeSeriesMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private static void WriteResultsToConsole(HistoricalTimeSeriesMessage responseMessage)
        {
            // copy results and display in console
            foreach (HistoricalTimeSeriesEntryMessage historicalItem in responseMessage.entry)
            {
                DateTime date = DateTime.MinValue;

                if (historicalItem.time.HasValue)
                    date = DateTimeConverter.FromUtcMillis(historicalItem.time.Value);

                Console.WriteLine("Date: " + date.ToShortDateString() +
                                  " Open: " + historicalItem.open +
                                  " High: " + historicalItem.high +
                                  " Low: " + historicalItem.low +
                                  " Close: " + historicalItem.close +
                                  " Isue: " + historicalItem.issue +
                                  " Redemtion: " + historicalItem.redemption +
                                  " Volume: " + historicalItem.volume);
            }
        }
    }
}
