﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;


namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * IndexCompositionSample
    * returns the vwd codes from the given index (e.g. 846900.ETR, CH0009980894)
    * - request via ISIN or VWDCode
    * - response has vwdCode,ISIN, instrument type and name property
    */
    public class IndexCompositionSample
    {
        public void TestIndexComposition(RoutingDataStore rds, MessageSerializer serializer)
        {
            /* Availability of function depends on contract and comes with additional costs. */

            // create the request object - possible using ISIN or vwdCode
            IndexCompositionRequestMessage requestMessage = new IndexCompositionRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword(),
                    sourceTarget = SourceTargetType.SOURCE_TARGET_DMC
                },
                addQuotes = false,
                isin = "DE0008469008",
                //vwdCode = "846900.ETR"
            };

            Console.WriteLine("Get Index composition by isin:" + requestMessage.isin);
            //Console.WriteLine("Get Index composition by vwdCode:" + requestMessage.vwdCode);

            // call the DMC backend
            Console.WriteLine("Calling IndexComposition function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.INDEX_COMPOSITION,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<IndexCompositionRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            IndexCompositionMessage responseMessage = serializer.Deserialize<IndexCompositionMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private static void WriteResultsToConsole(IndexCompositionMessage responseMessage)
        {
            //get and print results
            Console.WriteLine("RoutingDataStore response for IndexCompositionMessage : contained vwd codes: " + responseMessage.indexList.Count);
            foreach (IndexCompositionEntryMessage indexEntry in responseMessage.indexList)
            {
                Console.Write(" vwd Code: " + indexEntry.quotationIdentifiers.vwdCode + " || ");
                Console.Write(" Instrument type: " + indexEntry.quotationIdentifiers.instrumentType + " || ");
                Console.Write(" ISIN: " + indexEntry.quotationIdentifiers.isin + " || ");
                Console.Write(" Name: " + indexEntry.quotationIdentifiers.name + " || ");
                Console.WriteLine("\r\n___________________________________________________________________________________________________________");
            }
        }
    }
}
