﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;


namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
     * IntradayTimeSeriesSample
     * returns intraday data of an instrument (OHLCV) with respect to a certain aggregation.
     */
    public class IntradayTimeSeriesSample
    {
        private RoutingDataStore rds;
        private MessageSerializer serializer;

        public void TestIntradayTimeSeries(RoutingDataStore rds, MessageSerializer serializer)
        {
            this.rds = rds;
            this.serializer = serializer;

            // define parameters to use within the request
            DateTime start = DateTime.Now.AddDays(-2);
            DateTime end = DateTime.Now;

            // create the request object
            IntradayTimeSeriesRequestMessage intradayRequestMessage = new IntradayTimeSeriesRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                dateFrom = DateTimeConverter.Convert(ref start),
                dateTo = DateTimeConverter.Convert(ref end),
                instrument = "846900.ETR",
                withTrade = true,
                withAsk = false,
                withBid = false
            };

            // Get all Ticks. Without any period aggregation. The time range is limited for circa 1,2 Weeks in the past.
            Console.WriteLine("Intraday data aggregated by ticks. " + start.ToString() + " - " + end.ToString());
            intradayRequestMessage.aggregation = IntradayAggregationParam.Ticks;
            intradayRequestMessage.aggregationUnitValue = 1;
            GetIntradayResponseAndWriteOnConsole(intradayRequestMessage);

            // Get intraday data aggregated by minutes.
            Console.WriteLine("Intraday data aggregated by minutes. " + start.ToString() + " - " + end.ToString());
            intradayRequestMessage.aggregation = IntradayAggregationParam.Minutes;
            intradayRequestMessage.aggregationUnitValue = 1;
            GetIntradayResponseAndWriteOnConsole(intradayRequestMessage);

            // Get intraday data aggregated by 2 hours.
            Console.WriteLine("Intraday data aggregated by 2 hours. " + start.ToString() + " - " + end.ToString());
            intradayRequestMessage.aggregation = IntradayAggregationParam.Hours;
            intradayRequestMessage.aggregationUnitValue = 2;
            GetIntradayResponseAndWriteOnConsole(intradayRequestMessage);
        }

        // function to get intradayResponse and write it in the console
        private void GetIntradayResponseAndWriteOnConsole(IntradayTimeSeriesRequestMessage intradayRequestMessage)
        {
            // call the DMC backend
            Console.WriteLine("Calling Intraday function. Please, note that specific permissions might be needed by contract.");
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.INTRADAY,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<IntradayTimeSeriesRequestMessage>(OutputFormat.PROTO, intradayRequestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            IntradayTimeSeriesMessage responseMessage = serializer.Deserialize<IntradayTimeSeriesMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            // get and print data for Trades
            foreach (IntradayTimeSeriesEntryMessage tradeEntry in responseMessage.trade)
            {
                Console.Write("Trade - ");
                WriteResultsToConsole(tradeEntry);
            }
        }

        private static void WriteResultsToConsole(IntradayTimeSeriesEntryMessage entry)
        {
            if (entry.open.HasValue)
            {
                Console.Write("Open: " + entry.open.Value + " | ");
            }
            if (entry.high.HasValue)
            {
                Console.Write(" High: " + entry.high.Value + " | ");
            }
            if (entry.low.HasValue)
            {
                Console.Write(" Low: " + entry.low.Value + " | ");
            }
            if (entry.close.HasValue)
            {
                Console.Write(" Close: " + entry.close.Value + " | ");
            }
            if (entry.volume.HasValue)
            {
                Console.Write(" Volume: " + entry.volume.Value + " | ");
            }
            if (entry.time.HasValue)
            {
                DateTime time = DateTimeConverter.FromUtcMillis(entry.time.Value);
                Console.Write("Time: " + time + " | ");
            }
            Console.WriteLine("\r\n___________________________________________________________________________________________________________");
        }
    }
}
