﻿/**
 * Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
 *
 * This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
 * confidential. It may not be copied and/or distributed without the express permission of 
 * "Infront Financial Technology GmbH".
 */
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;


namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
     * QuotationSnapshotSample
     * - get quote snapshots via static data interface - NO push data!
     * - parameters to be set: vwd code and (optionally) field IDs
     */
    public class QuoteSnapshotSample
    {
        public void TestQuoteSnapshot(RoutingDataStore rds, MessageSerializer serializer)
        {
            var vwdCodeList = new List<string>();
            vwdCodeList.Add("846900.ETR");
            vwdCodeList.Add("HW3VDD.STG");

            // OR add instruments using a text file "SampleInstruments.txt" (included in Data folder)
            //var instrumentsFileName = Path.Combine(Environment.CurrentDirectory, "../../Data", "SampleInstruments.txt");
            //InstrumentList vwdCodeList = new InstrumentList();
            //vwdCodeList.AppendFromFile(instrumentsFileName);

            // All fieldIdentifiers can be displayed using FieldMap client sample
            // 25 (Date) 26 (Time) 57(ISIN) 58(Long security name) 59(Security name indicator) 47(Country) 48(Currency) 49(Short security name) 
            // 76(Security identification number) 78(IT identification code) 96(Symbol) 95(Strike price) 97(Exchange) 36(PrevDay's total volume)
            // 121(Close PrevDay's date) 122(PrevDay's close) 124(PrevDay's high) 128(PrevDay's low) 130(PrevDay's open)
            #region (Feldnamen auf Deutsch)
            // 25 (Datum) 26 (Zeit) 57(ISIN) 58(WP Name lang) 59(WP Name Zusatz) 47(Land) 48(Währung) 49(WP Name kurz) 
            // 76(WPK) 78(EDV Kürzel) 96(Symbol) 95(Basispreis) 97(Börse) 36(Umsatz gesamt Vortag)
            // 121(Schluss Vortagesdatum) 122(Schluss Vortag) 124(Tageshoch Vortag) 128(Tagestief Vortag) 130(Anfang Vortag)
            #endregion
            var fieldIdentifiersList = new PrintableList<string>()
                { "25", "26", "57", "58", "59", "47", "48", "49",
                  "76", "78", "96", "95", "97", "36",
                  "121", "122", "124", "128", "130"
                };

            /// create the request object
            QuotationDataRequestMessage requestMessage = new QuotationDataRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                vwdCodes = vwdCodeList,
                fieldIdentifiers = fieldIdentifiersList,
                addArbitrage = false,
                addArbitrageQuotes = false,
                addAssociatedQuotes = false,
                deltaTime = 0
            };

            // call the DMC backend
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.QUOTE_SNAPSHOT_V3,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<QuotationDataRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            QuotationDataListsMessage responseMessage = serializer.Deserialize<QuotationDataListsMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);
        }

        private static void WriteResultsToConsole(QuotationDataListsMessage responseMessage)
        {
            foreach (var quotation in responseMessage.quotationList)
            {
                //to extract a value for a field id, for example 58(WP Name lang) use following code part
                foreach (var fieldItem in quotation.fieldList)
                {
                    //addtitionally to the defined vwd field ids the response fieldList follows field ids with values:
                    //20008 : LastCloseDate, 20001 : LastClosePrice, 1774 : DataqualityType(RT: Real Time, D: Delay, EOD: End of Day)

                    // assign field value depending on the DataType
                    string fieldValue = "";
                    switch (fieldItem.fieldType)
                    {
                        case QuotationDataFieldType.DT_BOOLEAN:
                            if (fieldItem.boolValue != null) fieldValue = fieldItem.boolValue.Value.ToString();
                            break;
                        case QuotationDataFieldType.DT_DATETIME:
                            if (fieldItem.dateTimeValue.HasValue)
                            {
                                fieldValue = DateTimeConverter.FromUtcMillis(fieldItem.dateTimeValue.Value).ToString();
                            }
                            break;
                        case QuotationDataFieldType.DT_DOUBLE:
                            fieldValue = fieldItem.doubleValue.Value.ToString();
                            break;
                        case QuotationDataFieldType.DT_INT:
                            if (fieldItem.intValue != null) fieldValue = fieldItem.intValue.Value.ToString();
                            break;
                        case QuotationDataFieldType.DT_LONG:
                            if (fieldItem.longValue != null) fieldValue = fieldItem.longValue.Value.ToString();
                            break;
                        case QuotationDataFieldType.DT_STRING:
                            fieldValue = fieldItem.stringValue;
                            break;
                            // similarly you can extract data for types float, long e.t.c
                    }

                    Console.WriteLine("Instrument: " + quotation.vwdCode + " ID: " + fieldItem.id + " value: " + fieldValue);
                }
            }
        }
    }
}
