﻿/**
 * Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
 *
 * This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
 * confidential. It may not be copied and/or distributed without the express permission of
 * "Infront Financial Technology GmbH".
 */
using Gevasys.DataCore.Configuration;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.RequestData.ProtoBuf.V2.Retransmit;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using System;
using System.Collections.Generic;
using System.Threading;

namespace DataManagerConnect.SampleApplication.Samples.retransmit
{
    /**
     * RetransmitDataSample
     * request missed push data from the past (max the last 5 days). It returns a file download,
	 * but it seems to be a push data stream, the requested data comes in the same format as the push data.
     **/
    public class RetransmitDataSample : IAsyncCallback
    {
        bool retransmitCompleted = false;

        public void TestRetransmitData()
        {
            // create retransmit client
            var retransmitClient = new RetransmitSampleClient("RETRANSMIT-PUSH-CLIENT1");
            retransmitClient.Start();

            //create store
            RetransmitStore retransmitStore = ConfigurationHelper.CreateRetransmitStore(retransmitClient);

            if (retransmitStore == null)
            {
                Logger.Log.Error("INTERNAL ERROR: Failed to create retransmit store!");
                return;
            };

            // Set retransmit period for missing data for the last hours
            DateTime start = DateTime.Now.AddHours(-2);
            DateTime end = DateTime.Now.AddHours(-2).AddMinutes(5);

            // define symbols list
            List<string> symbols = new List<string>();
            symbols.Add("846900.ETR");

            // define exchanges list
            List<string> exchanges = new List<string>();
            exchanges.Add("FXVWD");

            // Define request entry with symbols list and time period
            RetransmitRequestEntryMessage entry = new RetransmitRequestEntryMessage
            {
                UTCStart = new DateTimeOffset(start).ToUnixTimeMilliseconds(),
                UTCEnd = new DateTimeOffset(end).ToUnixTimeMilliseconds(),
                //Symbols = symbols,
                Exchanges = exchanges
            };

            //create list with entry requests
            List<RetransmitRequestEntryMessage> requestEntryList = new List<RetransmitRequestEntryMessage>();
            requestEntryList.Add(entry);

            //Create instance of the request message for a retransmit
            RetransmitRequestMessage requestMessage = new RetransmitRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = ConfigurationHelper.Configuration.UserId,
                    password = ConfigurationHelper.Configuration.Password,
                    sourceTarget = SourceTargetType.SOURCE_TARGET_DMC
                },
                //Bandwith limitation in bytes per second
                maxBytesPerSecond = 0,
                requestID = 3,
                requestEntryList = requestEntryList
            };

            Console.WriteLine("Start Retransmit request...:" + start + " - " + end);

            //execute the retransmit request
            retransmitStore.RequestRetransmit(requestMessage, this);

            DateTime started = DateTime.Now;

            // wait until retransmit has finished
            while (!retransmitCompleted)
            {
                Thread.Sleep(1000);

                // timeout fuse
                if (DateTime.Now.Subtract(started).TotalSeconds > 15 * 60)
                {
                    retransmitCompleted = true;
                    Console.WriteLine("*** ABORTING BECAUSE OF TIMEOUT!");
                }

            }

            DateTime finished = DateTime.Now;

            Console.WriteLine("Started : " + started.ToLongTimeString());
            Console.WriteLine("Finished: " + finished.ToLongTimeString());
            StopRetransmit(retransmitClient, retransmitStore);
        }


        private static void StopRetransmit(RetransmitSampleClient retransmitClient, RetransmitStore retransmitStore)
        {
            retransmitClient.Stop();
            retransmitStore.Stop();
        }


        public void Completed(bool success, object state)
        {
            Console.WriteLine("*** Retransmit completed! Success=" + success);
            retransmitCompleted = true;

        }
    }
}

