﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using Vwd.Middleware.Common.DataModel.Base;
using Vwd.Middleware.Common.DataModel.Base.BaseClasses;
using MessageSerializer = Gevasys.DataCore.RequestData.ProtoBuf.V2.MessageSerializer;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
   /**
   * CorporateActionsSymbolSample
   * - get all corporate Actions for a symbol in the given period
   * - parameters to be set: start, end and symbols
   */
    public class CorporateActionsSymbolSample
    {
        public void TestCorporateActionsSymbol(RoutingDataStore rds, MessageSerializer serializer)
        {
            // define parameters to use within the request
            Timestamp start = Timestamp.ConvertToTimestamp(new DateTime(2015, 01, 01));
            Timestamp end = Timestamp.ConvertToTimestamp(DateTime.Now);
            List<string> symbols = new List<string>
            {
                "DB1_D.BXE4",
                "US5949181045"
            };

            // create the request object
            CorporateActionsRequest corporateActionsRequest = new CorporateActionsRequest
            {
                Header = new Header
                {
                    User = Authentication.GetUser(),
                    Password = Authentication.GetPassword()
                },
                Start = start,
                End = end,
                Symbols = symbols
            };

            // call the DMC backend
            Console.WriteLine("Calling CorporateActions function. Please, note that specific permissions might be needed by contract.");           
            var response = rds.CallFunction(StaticDataFunctions.GENERAL_CORPORATE_ACTIONS, 
                InputFormat.PROTO.ToString(), 
                OutputFormat.PROTO.ToString(),
                serializer.Serialize(OutputFormat.PROTO, corporateActionsRequest));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            CorporateActionsResponse responseMessage = serializer.Deserialize<CorporateActionsResponse>(InputFormat.PROTO, response.GetData());

            if (responseMessage.HasError)
            {
                Logger.Log.Warn("Error: " + responseMessage.Error.ErrorMessage);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            WriteResultsToConsole(responseMessage);                     
        }

        private void WriteResultsToConsole(CorporateActionsResponse responseMessage)
        {
            //get results list and print
            List<CorporateActionsInfoItem> resultList = responseMessage.ResultList;

            foreach (CorporateActionsInfoItem corporateAction in resultList)
            {
                InstrumentBaseData instrumenInfo = corporateAction.InstrumentInfo;
                Console.WriteLine("--------------------------------------" + corporateAction.symbol + "--------------------------------------");
                Console.WriteLine("ISIN: " + instrumenInfo.Isin + " - Name: " + instrumenInfo.Name + " -  TickerSymbol: " + instrumenInfo.TickerSymbol + " - Type: " + instrumenInfo.Type);
                Console.WriteLine("--------------------------------------------------------------------------------------");
                if (corporateAction.Actions != null)
                {
                    foreach (CorporateActionsInfo action in corporateAction.Actions)
                    {
                        Console.WriteLine("type: " + action.ActionType + " ,yield: " + action.Yield + " ,actionSum: " + action.ActionSum + " ,currency: " + action.Currency + " ,dateTime: " + action.Date.ToString());
                    }
                }
            }
        }
    }
}
