﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using System.Collections.Generic;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;

namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /**
    * IsinArbitrageSample
    * delivers all vwd-Codes for the given isin/wkn
    */
    public class IsinArbitrageSample
    {     
        public void TestIsinArbitrage(RoutingDataStore rds, MessageSerializer serializer)
        {
            List<string> isinList = new List<string>
            {
                "NL0000009827",
                "DE0005557508",
                "US2605661048",
                "BE0389555039",
                "FR0003500008"
            };
            List<string> instrumentList = new PrintableList<string>();
            instrumentList.AddRange(isinList);

            // create the request object
            ArbitrageRequestMessage requestMessage = new ArbitrageRequestMessage
            {
                header = new RequestHeaderMessage
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                // provide list of ISINs that vwd-internal listing codes
                instrumentList = instrumentList
            };

            // call the DMC backend
            StaticDataResponse response = rds.CallFunction(StaticDataFunctions.ISIN_ARBITRAGE_V2,
                InputFormat.PROTO.ToString(),
                OutputFormat.PROTO.ToString(),
                serializer.Serialize<ArbitrageRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return;
            }

            //get results
            ArbitrageListsMessage responseMessage = serializer.Deserialize<ArbitrageListsMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return;
            }

            GetResultsAndWriteToConsole(responseMessage);
        }

        private static void GetResultsAndWriteToConsole(ArbitrageListsMessage responseMessage)
        {
            ISINArbitrageResult isinArbitrageResult = new ISINArbitrageResult();

            isinArbitrageResult.ArbitrageList = new List<ArbitrageListMessage>();
            if (responseMessage.arbitrageLists != null)
            {
                isinArbitrageResult.ArbitrageList.AddRange(responseMessage.arbitrageLists);
            }

            isinArbitrageResult.NotFoundInstruments = new List<string>();
            if (responseMessage.notFoundInstruments != null)
            {
                isinArbitrageResult.NotFoundInstruments.AddRange(responseMessage.notFoundInstruments);
            }

            // print results to console
            foreach (ArbitrageListMessage arbitrageList in isinArbitrageResult.ArbitrageList)
            {
                foreach (ArbitrageMessage message in arbitrageList.list)
                {
                    Console.WriteLine("ISIN: " + arbitrageList.requestedInstrument + " , assigned vwd code:" + message.vwdCode +
                                               " , exchangeCodeVwd: " + message.exchangeCodeVwd + " , IsoCurrency: " + message.iso4217Currency);
                    Console.WriteLine("\r_______________________________________________________________________________________________________");
                }
            }
        }
    }
}

