﻿/**
* Copyright (C) 2020 by "Infront Financial Technology GmbH" - All Rights Reserved
*
* This file is part of/belongs to a project and/or product developed by "Infront Financial Technology GmbH".
*
* Unauthorized copying of this file, via any medium is strictly prohibited. Proprietary and
* confidential. It may not be copied and/or distributed without the express permission of 
* "Infront Financial Technology GmbH".
*/
using System;
using System.Collections.Generic;
using System.Linq;
using Gevasys.DataCore.Core.Logging;
using Gevasys.DataCore.RequestData.ProtoBuf.V2;
using Gevasys.DataCore.StaticData;
using Gevasys.DataCore.StaticData.Http;
using Gevasys.DataCore.StaticData.Util;
using Gevasys.DataCore.Webservice;
using ErrorCodes = Gevasys.DataCore.RequestData.ProtoBuf.V2.ErrorCodes;


namespace DataManagerConnect.SampleApplication.Samples.pull
{
    /** OptionFinderSampke
    *  returns the option which fulfill the filter
    * 
    *  Choose minimum one filter:
    * -by using different filters (e.g. vwdCodes, underlyingIsin and markets) the result should fulfill both filters
    * -by using an array (e.g. underlyingIsins) the result should fulfill one of the values
    **/
    public class OptionFinderSample
    {
        private RoutingDataStore rds;
        private MessageSerializer serializer;

        public void TestOptionFinder(RoutingDataStore rds, MessageSerializer serializer)
        {
            this.rds = rds;
            this.serializer = serializer;

            // Define parameters to use in filters (vwdCodes, underlyingIsins, markets)
            var vwdCodes = new List<string> { "850727.DTB.28_5.0R_1" };
            var underlyingIsins = new List<string> { "NL0000852564", "FR0000120271 " };
            var markets = new List<string> { "DTB" }; // examples for markets {"CBF", DTB"} 

            // Create and build option finder query using minimum 1 filter
            var optionFinderQuery = OptFinderQueryBuilder.NewQuery()
                                                   .FilterVwdCode(vwdCodes)
                                                   .FilterUnderylingIsins(underlyingIsins)
                                                   //.FilterMarket("DTB") //pay attention to permorfance
                                                   .Build();

            // get options list
            var options = GetOptionsByPaging(optionFinderQuery);

            if (options == null)
            {
                Logger.Log.Error("Error getting options. Please, check the specified filter for the query is correct.");
                return;
            }

            // get options list - first 1000 items
            foreach (FinderOPTListMessage optionInstrument in options.Take(1000))
            {
                if (optionInstrument == null)
                {
                    Logger.Log.Error("Result empty getting options.");
                    return;
                }

                PrintOptionInstrumentData(optionInstrument);
            }
        }

        #region GetOptionsByPaging - function organizing the paging through the option list
        private List<FinderOPTListMessage> GetOptionsByPaging(string query)
        {
            var options = new List<FinderOPTListMessage>();
            Console.WriteLine("GetOptionsByQuery: use paging in order to get complete list of options");

            bool hasMoreOptions = true;
            int optionsPerPage = 1000; // for paging
            int offset = 0; // for paging
            // < 50000 is not needed but if you get many instruments as result e.g. by using FilterMarket you must pay attention on system performance
            while (hasMoreOptions && offset < 50000)
            {
                var responseFromDMCBackend = GetOptionsFromDMCBackend(optionsPerPage, offset, query);

                if (responseFromDMCBackend == null || responseFromDMCBackend.finderList == null)
                {
                    Logger.Log.Error("GetOptionsFromDMCBackend returned no result(list is null or empty)");
                    return null;
                }

                options.AddRange(responseFromDMCBackend.finderList);

                int totalNumberAvailableOptions;
                int.TryParse(responseFromDMCBackend.total, out totalNumberAvailableOptions);
                offset += optionsPerPage;
                hasMoreOptions = offset < totalNumberAvailableOptions;
            }

            return options;
        }
        #endregion

        #region GetOptionsFromDMCBackend - function organizing the call to the OPT_FINDER proxy call
        private OPTFinderMessage GetOptionsFromDMCBackend(long count, long offset, string query)
        {
            // create request object
            OPTFinderRequestMessage requestMessage = new OPTFinderRequestMessage
            {
                header = new RequestHeaderMessage()
                {
                    user = Authentication.GetUser(),
                    password = Authentication.GetPassword()
                },
                query = query,
                count = count,
                offset = offset,
                sortBy = "symbol",  // "name" or "symbol" -- for OPRA: uses "symbol"
                @ascending = true
            };

            // call DMC Backend to get Options
             StaticDataResponse response = rds.CallFunction(StaticDataFunctions.OPT_FINDER,
                     InputFormat.PROTO.ToString(),
                     OutputFormat.PROTO.ToString(),
                     serializer.Serialize<OPTFinderRequestMessage>(OutputFormat.PROTO, requestMessage));

            if (response.GetData() == null)
            {
                Logger.Log.Error("There is no data for the request. Error: " + response.GetErrorMessage());
                Logger.Log.Error("Please, inform Infront support to check the server configuration if the error persists.");
                return null;
            }

            //get results
            OPTFinderMessage responseMessage = serializer.Deserialize<OPTFinderMessage>(InputFormat.PROTO, response.GetData());

            if (responseMessage.header.error_code != ErrorCodes.Ok)
            {
                Logger.Log.Warn("Error: " + responseMessage.header.error_message);
                Logger.Log.Warn("Error getting response data. Please, check if requested data is correct or if you are missing permissions to use this function.");
                return null;
            }

            var result = responseMessage;
            return result;
        }
        #endregion

        #region helper - PrintOptionInstrumentData
        private void PrintOptionInstrumentData(FinderOPTListMessage optionInstrument)
        {
            if (optionInstrument != null)
            {
                // Instrument specific data
                if (optionInstrument.instrumentData != null)
                {
                    Console.WriteLine("********************\r\nInstrument members: \r\n********************");
                    Console.WriteLine("Id  : " + optionInstrument.instrumentData.iid + " <<<");
                    Console.WriteLine("ISIN: " + optionInstrument.instrumentData.isin + " <<<");
                    Console.WriteLine("Name: " + optionInstrument.instrumentData.name + " <<<");
                    Console.WriteLine("Type: " + optionInstrument.instrumentData.type + " <<<");
                    Console.WriteLine("WKN : " + optionInstrument.instrumentData.wkn + " <<<");
                }
                // Quote data
                if (optionInstrument.quoteData != null)
                {
                    Console.WriteLine("********************\r\nQuote data members: \r\n********************");
                    Console.WriteLine("Vwdcode   : " + optionInstrument.quoteData.vwdCode + " <<<");
                    Console.WriteLine("Market@Vwd: " + optionInstrument.quoteData.marketVwd + " <<<");
                    Console.WriteLine("    - Name: " + optionInstrument.quoteData.marketName + " <<<");
                    Console.WriteLine("Curr ISO  : " + optionInstrument.quoteData.currencyIso + " <<<");
                    Console.WriteLine("WKN       : " + optionInstrument.quoteData.mmwkn + " <<<");
                    Console.WriteLine("ID        : " + optionInstrument.quoteData.qid + " <<<");
                }
                // more instrument / option attributes
                Console.WriteLine("***********************\r\nInstrument attributes: \r\n***********************");
                if (optionInstrument.underlyingIsin != null)
                    Console.WriteLine("Underlying ISIN: " + optionInstrument.underlyingIsin + " <<<");
                if (optionInstrument.underlyingName != null)
                    Console.WriteLine("Underlying Name: " + optionInstrument.underlyingName + " <<<");
                if (optionInstrument.optionType != null)
                    Console.WriteLine("Option Typ: " + optionInstrument.optionType + " <<<");
                if (optionInstrument.optionCategory != null)
                    Console.WriteLine("Option cat: " + optionInstrument.optionCategory + " <<<");
                if (optionInstrument.expirationDate != null)
                    Console.WriteLine("Expiration date: " + DateTimeConverter.FromUtcMillis(optionInstrument.expirationDate.Value) + " <<<");
                if (optionInstrument.tradingMonth != null)
                    Console.WriteLine("Trading month: " + optionInstrument.tradingMonth + " <<<");
                if (optionInstrument.contractSize != null)
                    Console.WriteLine("Contract size: " + optionInstrument.contractSize + " <<<");
                if (optionInstrument.contractValue != null)
                    Console.WriteLine("Contract value: " + optionInstrument.contractValue + " <<<");
                if (optionInstrument.contractValueCalculated != null)
                    Console.WriteLine("Contract value calc: " + optionInstrument.contractValueCalculated + " <<<");
                if (optionInstrument.generationNumber != null)
                    Console.WriteLine("Generation no: " + optionInstrument.generationNumber + " <<<");
                if (optionInstrument.versionNumber != null)
                    Console.WriteLine("Version no: " + optionInstrument.versionNumber + " <<<");

                // Price data for instrument
                Console.WriteLine("***********************\r\nInstrument price data: \r\n***********************");
                if (optionInstrument.bid != null)
                    Console.WriteLine("Bid: " + optionInstrument.bid + " <<<");
                if (optionInstrument.ask != null)
                    Console.WriteLine("Ask: " + optionInstrument.ask + " <<<");
                if (optionInstrument.price != null)
                    Console.WriteLine("Price: " + optionInstrument.price + " <<<");
                if (optionInstrument.date != null)
                    Console.WriteLine("last traded date: " + DateTimeConverter.FromUtcMillis(optionInstrument.date.Value) + " <<<");
            }
        }
        #endregion
    }
}
